package ch.codeblock.qrinvoice.model;

import java.util.Objects;

/**
 * <p>From the specification v1.0</p>
 * <table border="1" summary="Excerpt from the specification">
 * <tr><th>Language</th><th>General Definition</th><th>Field Definition</th></tr>
 * <tr><td>EN</td><td>Payment reference</td><td>Mandatory data group</td></tr>
 * <tr><td>DE</td><td>Zahlungsreferenz</td><td>Obligatorische Datengruppe</td></tr>
 * <tr><td>FR</td><td>Référence de paiement</td><td>Groupe de données obligatoire</td></tr>
 * <tr><td>IT</td><td>Riferimento dl pagamento</td><td>Gruppo di dati obbligatorio</td></tr>
 * </table>
 * <p>Data Structure Element</p>
 * <pre>
 * QRCH
 * +RmtInf
 * </pre>
 */
public class PaymentReference {
    /**
     * <p>From the specification v1.0</p>
     * <table border="1" summary="Excerpt from the specification">
     * <tr><th>Language</th><th>General Definition</th><th>Field Definition</th></tr>
     * <tr><td>EN</td><td>Reference type<br>Reference type (QR, ISO)<br>The following codes are permitted:<br>QRR – QR reference<br>SCOR – Creditor Reference (ISO 11649)<br>NON – without reference</td><td>Maximum four characters, alphanumeric; with the use of a QR-IBAN must contain the QRR code or SCOR.</td></tr>
     * <tr><td>DE</td><td>Referenztyp<br>Referenztyp (QR, ISO)<br>Die folgenden Codes sind zugelassen:<br>QRR – QR-Referenz<br>SCOR – Creditor Reference (ISO 11649)<br>NON – ohne Referenz</td><td>Maximal vier Zeichen, alphanumerisch; muss bei Verwendung einer QR-IBAN den Code QRR oder SCOR enthalten.</td></tr>
     * <tr><td>FR</td><td>Type de référence<br>Type de référence (QR, ISO)<br>Les codes suivants sont admis:<br>QRR – Référence QR<br>SCOR – Creditor Reference (ISO 11649) NON – sans référence</td><td>Quatre caractères au maximum; en cas d'utilisation d'un QR-IBAN, doit contenir le code QRR ou SCOR.</td></tr>
     * <tr><td>IT</td><td>Tipo di riferimento<br>Tipo di riferimento (QR, ISO)<br>Sono ammessi i seguenti codici:<br>QRR – Riferimento-QR<br>SCOR – Creditor Reference (ISO 11649) NON – Senza riferimento</td><td>Massimo quattro caratteri, alfanumerici<br>In caso di utilizzo di un QR-IBAN, deve contenere il codice QRR o ISO.</td></tr>
     * </table>
     * <p>Status: {@link Status#MANDATORY}</p>
     * <p>Data Structure Element</p>
     * <pre>
     * QRCH
     * +RmtInf
     * ++Tp
     * </pre>
     * 
     * @see ReferenceType#QR_REFERENCE
     * @see ReferenceType#CREDITOR_REFERENCE
     * @see ReferenceType#WITHOUT_REFERENCE
     */
    private ReferenceType referenceType;
    /**
     * <p>From the specification v1.0</p>
     * <table border="1" summary="Excerpt from the specification">
     * <tr><th>Language</th><th>General Definition</th><th>Field Definition</th></tr>
     * <tr><td>EN</td><td>Reference<br>Reference number<br>Structured payment reference<br>Note: The reference is either a QR reference or a Creditor Reference (ISO 11649)</td><td>Maximum 27 characters, alphanumeric; must be filled if a QR-IBAN is used.<br><br>QR reference: 27 characters, numeric, check sum calculation according to Modulo 10 recursive (27th position of the reference) Creditor Reference (ISO 11649): up to 25 characters, alphanumeric. The element may not be filled for the NON reference type.</td></tr>
     * <tr><td>DE</td><td>Referenz<br>Referenznummer<br>Strukturierte Zahlungsreferenz<br>Anmerkung: Die Referenz ist entweder eine QR-Referenz oder Creditor Reference (ISO 11649)</td><td>Maximal 27 Zeichen, alphanumerisch; muss bei Verwendung einer QR-IBAN befüllt werden.<br>QR-Referenz: 27 Zeichen, numerisch; Prüfzifferberechnung nach Modulo 10 rekursiv (27ste Stelle der Referenz).<br>Creditor Reference (ISO 11649): bis 25 Zeichen, alphanumerisch.<br>Für den Referenztyp NON darf das Element nicht befüllt werden.</td></tr>
     * <tr><td>FR</td><td>Référence<br>Numéro de référence<br>Référence de paiement structurée<br>Remarque: La référence est soit une référence QR, soit une Creditor Reference (ISO 11649)</td><td>27 caractères alphanumériques au maximum; doit être rempli en cas d'utilisation d'un QR-IBAN.<br>Référence QR: 27 caractères numériques, calcul du chiffre de contrôle selon modulo 10 récursif (27e position de la référence). Creditor Reference (ISO 11649): jusqu'à 25 caractères alphanu- mériques.<br>L'élément ne doit pas être rempli pour le type de référence NON.</td></tr>
     * <tr><td>IT</td><td>Riferimento<br>Numero di riferimento<br>Riferimento strutturato del pagamento<br>Nota: il riferimento consiste in un riferimento-QR o ISO Creditor Reference (ISO 11649)</td><td>Massimo 27 caratteri, alfanumerici. In caso di utilizzo di un QR- IBAN deve esser compilato.<br>Riferimento-QR: 27 caratteri, numerici, calcolo della check digit secondo il modulo 10 ricorsivo (27° carattere del riferimento) SCOR – Creditor Reference (ISO 11649): fino a 25 caratteri, alfanumerici<br>Per il tipo di riferimento NON, l’elemento non deve essere compilato.</td></tr>
     * </table>
     * <p>Status: {@link Status#OPTIONAL}</p>
     * <p>Data Structure Element</p>
     * <pre>
     * QRCH
     * +RmtInf
     * ++Ref
     * </pre>
     */
    private String reference;
    /**
     * <p>From the specification v1.0</p>
     * <table border="1" summary="Excerpt from the specification">
     * <tr><th>Language</th><th>General Definition</th><th>Field Definition</th></tr>
     * <tr><td>EN</td><td>Unstructured message<br>Additional information<br>Additional information can be used for the scheme with message and scheme with structured reference to provide additional information to the biller. For the transmission of structured additional information to the debtor, the details in the ""creditor’s structured information"" section must be adhered to.</td><td>Maximum 140 characters</td></tr>
     * <tr><td>DE</td><td>Unstrukturierte Mitteilung<br>Zusätzliche Informationen<br>Zusätzliche Informationen können beim Verfahren mit Mitteilung und beim Verfahren mit strukturierter Referenz für zusätzliche Informationen an den Rechnungssteller verwendet werden.<br>Für die Übermittlung von strukturierten Zusatzinformationen an den Zahlungspflichtigen müssen die Angaben im Kapitel «Strukturierte Informationen des Rechnungsstellers» eingehalten werden.</td><td>Maximal 140 Zeichen</td></tr>
     * <tr><td>FR</td><td>Communication non structurée<br>Informations supplémentaires<br>Des informations supplémentaires peuvent être utilisées dans la procédure avec communication et dans celle avec référence structurée en vue d'informations supplémentaires à l'émetteur de la facture.<br>Les données prévues dans le chapitre «Informations structurées de l'émetteur de factures» doivent être respectées dans la transmission d'informations sup- plémentaires structurées.</td><td>140 caractères au maximum</td></tr>
     * <tr><td>IT</td><td>Messaggio non strutturato<br>Informazioni supplementari<br>Le informazioni supplementari possono essere usate nel<br>processo con messaggio e nel processo con riferimento strutturato per informazioni aggiuntive del mittente della fattura.<br>Per la trasmissione di informazioni supplementari strutturate al debitore è necessario rispettare le indicazioni del Capitolo «Informazioni strutturate del mittente della fattura».</td><td>Massimo 140 caratteri</td></tr>
     * </table>
     * <p>Status: {@link Status#OPTIONAL}</p>
     * <p>Data Structure Element</p>
     * <pre>
     * QRCH
     * +RmtInf
     * ++Ustrd
     * </pre>
     */
    private String unstructuredMessage;

    public ReferenceType getReferenceType() {
        return referenceType;
    }

    public void setReferenceType(final ReferenceType referenceType) {
        this.referenceType = referenceType;
    }

    public String getReference() {
        return reference;
    }

    public void setReference(final String reference) {
        this.reference = reference;
    }

    public String getUnstructuredMessage() {
        return unstructuredMessage;
    }

    public void setUnstructuredMessage(final String unstructuredMessage) {
        this.unstructuredMessage = unstructuredMessage;
    }

    public enum ReferenceType {
        /**
         * <p>QRR – QR reference</p>
         * <p>
         * <p>From the specification v1.0</p>
         * <table border="1" summary="Excerpt from the specification">
         * <tr><th>Language</th><th>Description</th></tr>
         * <tr><td>EN</td><td>The biller's structured reference corresponds to the former ISR reference number.</td></tr>
         * <tr><td>DE</td><td>Die strukturierte Referenz des Rechnungsstellers entspricht der ehemaligen ESR Referenznummer.</td></tr>
         * <tr><td>FR</td><td>Référence structurée de l'émetteur de factures, correspondant à l'ancien numéro de référence BVR.</td></tr>
         * <tr><td>IT</td><td>Riferimento strutturato del mittente della fattura, corrispondente all’ex numero di riferimento della polizza di versamento.</td></tr>
         * </table>
         *
         * @see ch.codeblock.qrinvoice.model.PaymentReference#referenceType
         */

        QR_REFERENCE("QRR"),
        /**
         * <p>SCOR – Creditor Reference (ISO 11649)</p>
         * <p>
         * <p>From the specification v1.0</p>
         * <table border="1" summary="Excerpt from the specification">
         * <tr><th>Language</th><th>Description</th></tr>
         * <tr><td>EN</td><td>Creditor Reference according to the ISO 11649 standard.</td></tr>
         * <tr><td>DE</td><td>Creditor Reference gemäss ISO-11649-Standard.</td></tr>
         * <tr><td>FR</td><td>Creditor Reference selon la norme ISO 11649.</td></tr>
         * <tr><td>IT</td><td>Riferimento strutturato al creditore in conformità allo standard ISO 11649.</td></tr>
         * </table>
         *
         * @see ch.codeblock.qrinvoice.model.PaymentReference#referenceType
         */
        CREDITOR_REFERENCE("SCOR"),
        /**
         * <p>NON – without reference</p>
         * 
         * <p>From the specification v1.0</p>
         * <table border="1" summary="Excerpt from the specification">
         * <tr><th>Language</th><th>Description</th></tr>
         * <tr><td>EN</td><td></td></tr>
         * <tr><td>DE</td><td></td></tr>
         * <tr><td>FR</td><td></td></tr>
         * <tr><td>IT</td><td></td></tr>
         * </table>
         *
         * @see ch.codeblock.qrinvoice.model.PaymentReference#referenceType
         */
        WITHOUT_REFERENCE("NON");

        private final String referenceTypeCode;

        ReferenceType(final String referenceTypeCode) {
            this.referenceTypeCode = referenceTypeCode;
        }

        public String getReferenceTypeCode() {
            return referenceTypeCode;
        }

        public static ReferenceType parse(final String referenceTypeCode) {
            for (final ReferenceType referenceType : values()) {
                if (referenceType.getReferenceTypeCode().equals(referenceTypeCode)) {
                    return referenceType;
                }
            }
            throw new ParseException("Invalid reference type '" + referenceTypeCode + "' given");
        }

        @Override
        public String toString() {
            return "ReferenceType{" +
                    "referenceTypeCode='" + referenceTypeCode + '\'' +
                    "} " + super.toString();
        }
    }

    @Override
    public String toString() {
        return "PaymentReference{" +
                "referenceType=" + referenceType +
                ", reference='" + reference + '\'' +
                ", unstructuredMessage='" + unstructuredMessage + '\'' +
                '}';
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        final PaymentReference that = (PaymentReference) o;
        return referenceType == that.referenceType &&
                Objects.equals(reference, that.reference) &&
                Objects.equals(unstructuredMessage, that.unstructuredMessage);
    }

    @Override
    public int hashCode() {
        return Objects.hash(referenceType, reference, unstructuredMessage);
    }
}
