package ch.codeblock.qrinvoice.model.mapper;

import ch.codeblock.qrinvoice.model.*;
import ch.codeblock.qrinvoice.util.DecimalFormatFactory;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.ParsePosition;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Currency;

public class SwissPaymentsCodeToModelMapper {
    public QrInvoice map(SwissPaymentsCode spc) {
        QrInvoice qrInvoice = new QrInvoice();
                
        qrInvoice.setHeader(mapHeader(spc));
        qrInvoice.setCreditorInformation(mapCreditorInformation(spc));
        qrInvoice.setUltimateCreditor(mapUltimateCreditor(spc));
        qrInvoice.setPaymentAmountInformation(mapPaymentAmountInformation(spc));
        qrInvoice.setUltimateDebtor(mapUltimateDebtor( spc));
        qrInvoice.setPaymentReference(mapPaymentReference(spc));
        qrInvoice.setAlternativeSchemes(mapAlternativeSchemes(spc));

        return qrInvoice;
    }


    private UltimateDebtor mapUltimateDebtor(final SwissPaymentsCode spc) {
        final UltimateDebtor ultimateDebtor = new UltimateDebtor();
        ultimateDebtor.setName(spc.getUdName());
        ultimateDebtor.setStreetName(spc.getUdStrtNm());
        ultimateDebtor.setHouseNumber(spc.getUdBldgNb());
        ultimateDebtor.setPostalCode(spc.getUdPstCd());
        ultimateDebtor.setCity(spc.getUdTwnNm());
        ultimateDebtor.setCountry(spc.getUdCtry());
        return ultimateDebtor;
    }

    private Header mapHeader(final SwissPaymentsCode spc) {
        final Header header = new Header();
        header.setQrType(spc.getQrType());
        header.setVersion(Short.parseShort(spc.getVersion()));
        header.setCodingType(Byte.parseByte(spc.getCoding()));
        return header;
    }

    private CreditorInformation mapCreditorInformation(final SwissPaymentsCode spc) {
        final CreditorInformation creditorInformation = new CreditorInformation();
        creditorInformation.setIban(spc.getIban());
        creditorInformation.setCreditor(mapCreditor(spc));
        return creditorInformation;
    }

    private Creditor mapCreditor(final SwissPaymentsCode spc) {
        final Creditor creditor = new Creditor();
        creditor.setName( spc.getCrName());
        creditor.setStreetName(spc.getCrStrtNm());
        creditor.setHouseNumber(spc.getCrBldgNb());
        creditor.setPostalCode(spc.getCrPstCd());
        creditor.setCity(spc.getCrTwnNm());
        creditor.setCountry(spc.getCrCtry());
        return creditor;
    }

    private UltimateCreditor mapUltimateCreditor(final SwissPaymentsCode spc) {
        final UltimateCreditor ultimateCreditor = new UltimateCreditor();
        ultimateCreditor.setName(spc.getUcrName());
        ultimateCreditor.setStreetName(spc.getUcrStrtNm());
        ultimateCreditor.setHouseNumber(spc.getUcrBldgNb());
        ultimateCreditor.setPostalCode(spc.getUcrPstCd());
        ultimateCreditor.setCity(spc.getUcrTwnNm());
        ultimateCreditor.setCountry(spc.getUcrCtry());
        return ultimateCreditor;
    }
    private PaymentReference mapPaymentReference(final SwissPaymentsCode spc) {
        final PaymentReference paymentReference = new PaymentReference();
        paymentReference.setReferenceType(PaymentReference.ReferenceType.parse(spc.getTp()));
        paymentReference.setReference(spc.getRef());
        paymentReference.setUnstructuredMessage(spc.getUstrd());
        return paymentReference;
    }

    private PaymentAmountInformation mapPaymentAmountInformation(final SwissPaymentsCode spc) {
        final PaymentAmountInformation paymentAmountInformation = new PaymentAmountInformation();
        final String amt = spc.getAmt();
        if(amt != null) {
            final DecimalFormat decimalFormat = DecimalFormatFactory.createQrCodeAmountFormat();
            decimalFormat.setParseBigDecimal(true);
            paymentAmountInformation.setAmount((BigDecimal) decimalFormat.parseObject(amt, new ParsePosition(0)));
        } else {
            paymentAmountInformation.setAmount(null);
        }

        paymentAmountInformation.setCurrency(Currency.getInstance(spc.getCcy()));

        final String date = spc.getReqdExctnDt();
        if(date != null) {
            paymentAmountInformation.setDate(LocalDate.parse(date, DateTimeFormatter.ISO_DATE));
        } else {
            paymentAmountInformation.setDate(null);
        }
        return paymentAmountInformation;
    }


    private AlternativeSchemes mapAlternativeSchemes(final SwissPaymentsCode spc) {
        final AlternativeSchemes alternativeSchemes = new AlternativeSchemes();
        alternativeSchemes.setAlternativeSchemeParameters(spc.getAltPmts());
        return alternativeSchemes;
    }

}
