package ch.codeblock.qrinvoice.model.validation;

import ch.codeblock.qrinvoice.model.Creditor;
import ch.codeblock.qrinvoice.model.UltimateCreditor;
import ch.codeblock.qrinvoice.model.UltimateDebtor;
import ch.codeblock.qrinvoice.model.interfaces.Address;

import static ch.codeblock.qrinvoice.model.validation.ValidationUtils.*;
import static ch.codeblock.qrinvoice.util.CountryUtils.isValidIsoCode;

public class AddressValidator {
    public void validate(final Creditor creditor, final ValidationErrors errors) {
        validateAddress("creditorinformation.creditor", creditor, true, errors);
    }

    public void validate(final UltimateCreditor ultimateCreditor, final ValidationErrors errors) {
        validateAddress("ultimateCreditor", ultimateCreditor, false, errors);
    }

    public void validate(final UltimateDebtor ultimateDebtor, final ValidationErrors errors) {
        validateAddress("ultimateDebtor", ultimateDebtor, false, errors);
    }

    private void validateAddress(final String baseDataPath, final Address address, final boolean mandatoryGroup, final ValidationErrors errors) {
        if (address == null) {
            if (mandatoryGroup) {
                errors.addError(baseDataPath, null, "validation.error.address.group");
            }
            return;
        }
        
        // @formatter:off
        // lengths
        validateLength        (address.getName(),        1, 70, (value) -> errors.addError(baseDataPath, "name",        value, "validation.error.address.name"));
        validateOptionalLength(address.getStreetName(),  1, 70, (value) -> errors.addError(baseDataPath, "streetName",  value, "validation.error.address.streetName"));
        validateOptionalLength(address.getHouseNumber(), 1, 16, (value) -> errors.addError(baseDataPath, "houseNumber", value, "validation.error.address.houseNumber"));
        validateLength        (address.getPostalCode(),  1, 16, (value) -> errors.addError(baseDataPath, "postalCode",  value, "validation.error.address.postalCode"));
        validateLength        (address.getCity(),        1, 35, (value) -> errors.addError(baseDataPath, "city",        value, "validation.error.address.city"));
        validateTrue(address.getCountry(), 
                isValidIsoCode(address.getCountry()),           (value) -> errors.addError(baseDataPath, "country",     value, "validation.error.address.country"));
        
        // characters
        validateCharacters    (address.getName(),               (value) -> errors.addError(baseDataPath, "name",        value, "validation.error.invalidCharacters"));
        validateCharacters    (address.getStreetName(),         (value) -> errors.addError(baseDataPath, "streetName",  value, "validation.error.invalidCharacters"));
        validateCharacters    (address.getHouseNumber(),        (value) -> errors.addError(baseDataPath, "houseNumber", value, "validation.error.invalidCharacters"));
        validateCharacters    (address.getPostalCode(),         (value) -> errors.addError(baseDataPath, "postalCode",  value, "validation.error.invalidCharacters"));
        validateCharacters    (address.getCity(),               (value) -> errors.addError(baseDataPath, "city",        value, "validation.error.invalidCharacters"));
        // @formatter:on 
    }

}
