package ch.codeblock.qrinvoice.model.validation;

import ch.codeblock.qrinvoice.NotYetImplementedException;
import ch.codeblock.qrinvoice.QrInvoiceSpec;
import ch.codeblock.qrinvoice.model.*;
import ch.codeblock.qrinvoice.util.CollectionUtils;
import ch.codeblock.qrinvoice.util.CreditorReferenceUtils;
import ch.codeblock.qrinvoice.util.IbanUtils;
import ch.codeblock.qrinvoice.util.QRReferenceUtils;

import java.math.BigDecimal;
import java.util.Currency;
import java.util.List;

import static ch.codeblock.qrinvoice.QrInvoiceSpec.*;
import static ch.codeblock.qrinvoice.model.validation.ValidationUtils.*;

public class QrInvoiceValidator {
    private final AddressValidator addressValidator = new AddressValidator();

    public ValidationErrors validate(QrInvoice qrInvoice) {
        final ValidationErrors errors = new ValidationErrors();

        validate(qrInvoice.getHeader(), errors);
        validate(qrInvoice.getCreditorInformation(), errors);
        validate(qrInvoice.getUltimateCreditor(), errors);
        validate(qrInvoice.getUltimateDebtor(), errors);
        validate(qrInvoice.getPaymentAmountInformation(), errors);
        validate(qrInvoice.getPaymentReference(), errors);
        validate(qrInvoice.getAlternativeSchemes(), errors);

        validateCrossDependentElements(qrInvoice, errors);
        
        return errors;
    }

    private void validate(final Header header, final ValidationErrors errors) {
        final byte codingType = header.getCodingType();
        validateTrue(codingType, codingType == QrInvoiceSpec.CODING_TPYE, (value) -> errors.addError("header", "codingType", codingType, "validation.error.codingType"));

        final short version = header.getVersion();
        validateTrue(version, version == Short.parseShort(QrInvoiceSpec.VERSION), (value) -> errors.addError("header", "version", version, "validation.error.version"));

        final String qrType = header.getQrType();
        validateTrue(qrType, QrInvoiceSpec.QR_TYPE.equals(qrType), (value) -> errors.addError("header", "qrType", qrType, "validation.error.qrType"));
    }

    private void validate(final CreditorInformation creditorInformation, final ValidationErrors errors) {
        addressValidator.validate(creditorInformation.getCreditor(), errors);

        final String iban = creditorInformation.getIban();
        validateTrue(iban, IbanUtils.isValidIBAN(iban, true), (value) -> errors.addError("creditorInformation", "iban", value, "validation.error.iban"));
    }

    private void validate(final UltimateCreditor ultimateCreditor, final ValidationErrors errors) {
        addressValidator.validate(ultimateCreditor, errors);
    }
    
    private void validate(final UltimateDebtor ultimateDebtor, final ValidationErrors errors) {
        addressValidator.validate(ultimateDebtor, errors);
    }

    private void validate(final PaymentAmountInformation paymentAmountInformation, final ValidationErrors errors) {
        final BigDecimal amount = paymentAmountInformation.getAmount();
        validateRange(amount, AMOUNT_MIN, AMOUNT_MAX, (value) -> errors.addError("paymentAmountInformation", "amount", amount, "validation.error.amount"));

        final Currency currency = paymentAmountInformation.getCurrency();
        validateTrue(currency, SUPPORTED_CURRENCIES.contains(currency), (value) -> errors.addError("paymentAmountInformation", "currency", value, "validation.error.currency"));
    }

    private void validate(final PaymentReference paymentReference, final ValidationErrors errors) {
        final PaymentReference.ReferenceType referenceType = paymentReference.getReferenceType();
        validateTrue(referenceType, referenceType !=null, (value) -> errors.addError("paymentReference", "referenceType", value, "validation.error.referenceType"));
        
        final String reference  = paymentReference.getReference();
        validateCharacters(reference, (value) -> errors.addError("paymentReference", "reference", value, "validation.error.invalidCharacters"));
        if(referenceType != null) {
            switch (referenceType) {
                case QR_REFERENCE:
                    // QR reference: 27 characters, numeric, check sum calculation according to Modulo 10 recursive (27th position of the reference)
                    validateTrue(reference, QRReferenceUtils.isValid(reference), (value) -> errors.addError("paymentReference", "reference", value, "validation.error.reference", "validation.error.reference.QRR"));
                    break;
                case CREDITOR_REFERENCE:
                    // Creditor Reference (ISO 11649): up to 25 characters, alphanumeric
                    validateTrue(reference, CreditorReferenceUtils.isValid(reference), (value) -> errors.addError("paymentReference", "reference", value, "validation.error.reference", "validation.error.reference.SCOR"));
                    break;
                case WITHOUT_REFERENCE:
                    // The element may not be filled for the NON reference type.
                    validateEmpty(reference, (value) -> errors.addError("paymentReference", "reference", value, "validation.error.reference", "validation.error.reference.NON"));
                    break;
                default:
                    throw new NotYetImplementedException("ReferenceType '" + reference + "' is not yet implemented");
            }
        }

        final String unstructuredMessage = paymentReference.getUnstructuredMessage();
        validateOptionalLength(unstructuredMessage, 0, 140, (value) -> errors.addError("paymentReference", "unstructuredMessage", value, "validation.error.paymentReference.unstructuredMessage"));
        validateCharacters(unstructuredMessage, (value) -> errors.addError("paymentReference", "unstructuredMessage", value, "validation.error.invalidCharacters"));
    }
    
    private void validate(final AlternativeSchemes alternativeSchemes, final ValidationErrors errors) {
        if (alternativeSchemes != null) {
            final List<String> alternativeSchemeParameters = alternativeSchemes.getAlternativeSchemeParameters();
            final int size = CollectionUtils.size(alternativeSchemeParameters);
            if(size > MAX_ALT_PMT) {
                errors.addError("alternativeSchemes", "alternativeSchemeParameters", size, "validation.error.alternativeSchemes.alternativeSchemeParameters.size");
            }
            if(size > 0) {
                for (final String param : alternativeSchemeParameters) {
                    validateOptionalLength(param, 0, 100, (value) -> errors.addError("alternativeSchemes", "alternativeSchemeParameters", value, "validation.error.alternativeSchemes.alternativeSchemeParameter.length"));
                    validateCharacters(param, (value) -> errors.addError("alternativeSchemes", "alternativeSchemeParameters", value, "validation.error.invalidCharacters"));
                }
            }
        }
    }
    
    private void validateCrossDependentElements(final QrInvoice qrInvoice, final ValidationErrors errors) {
        final String iban = qrInvoice.getCreditorInformation().getIban();
        if(IbanUtils.isQrIBAN(iban)) {
            // with the use of a QR-IBAN must contain the QRR code or SCOR.
            final PaymentReference.ReferenceType referenceType = qrInvoice.getPaymentReference().getReferenceType();
            if(referenceType != null) {
                switch (referenceType) {
                    case QR_REFERENCE:
                    case CREDITOR_REFERENCE:
                        // in this case, everything is fine, the presence of the reference code is check in #validate(PaymentReference, ValidationErrors)
                        return;
                }
            }
            errors.addError("paymentReference", "referenceType", referenceType, "validation.error.referenceType.qrIban");
        }

    } 


}
