package ch.codeblock.qrinvoice.model.validation;

import ch.codeblock.qrinvoice.util.StringUtils;

import java.math.BigDecimal;
import java.util.function.Consumer;

import static ch.codeblock.qrinvoice.util.EncodingUtils.isValidEncoding;

public class ValidationUtils {
    // CharSequence
    public static void validateLength(final CharSequence str, final int min, final int max, final Consumer<CharSequence> validationErrorCallback) {
        validateLength(str, min, max, false, validationErrorCallback);
    }
    public static void validateOptionalLength(final CharSequence str, final int min, final int max, final Consumer<CharSequence> validationErrorCallback) {
        validateLength(str, min, max, true, validationErrorCallback);
    }
    
    private static void validateLength(final CharSequence str, final int min, final int max, final boolean optional, final Consumer<CharSequence> validationErrorCallback) {
        if(!validateLength(str, min, max, optional)) {
            validationErrorCallback.accept(str);
        }
    }
    
    public static boolean validateLength(final CharSequence str, final int min, final int max, final boolean optional) {
        // consider empty string only if min is greater than 0
        if (str == null || (str.length() == 0 && min > 0)) {
            return optional;
        }

        final int length = str.length();
        return (min <= length && length <= max);
    }

    public static void validateEmpty(final String value, final Consumer<String> validationErrorCallback) {
        if(StringUtils.isNotEmpty(value)) {
            validationErrorCallback.accept(value);
        }
    }
    
    public static void validateCharacters(final String value,final Consumer<CharSequence> validationErrorCallback) {
        if(StringUtils.isNotEmpty(value) && !isValidEncoding(value)) {
            validationErrorCallback.accept(value);
        }
    }
    
    // BigDecimal range
    public static void validateRange(final BigDecimal number, final BigDecimal min, final BigDecimal max, final Consumer<BigDecimal> validationErrorCallback) {
        validateRange(number, min, max, false, validationErrorCallback);
    }
    public static void validateOptionalRange(final BigDecimal number, final BigDecimal min, final BigDecimal max, final Consumer<BigDecimal> validationErrorCallback) {
        validateRange(number, min, max, true, validationErrorCallback);
    }

    private static void validateRange(final BigDecimal number, final BigDecimal min, final BigDecimal max, final boolean optional, final Consumer<BigDecimal> validationErrorCallback) {
        if(!validateRange(number, min, max, optional)) {
            validationErrorCallback.accept(number);
        }
    }
    
    public static boolean validateRange(final BigDecimal number, final BigDecimal min, final BigDecimal max, final boolean optional) {
        // consider empty string only if min is greater than 0
        if (number == null) {
            return optional;
        }

        final boolean greaterOrEqualThanMin = number.compareTo(min) >= 0;
        final boolean lessOrEqualThanMax = number.compareTo(max) <= 0;
        return greaterOrEqualThanMin && lessOrEqualThanMax;
    }

    public static <T> void validateTrue(final T value, boolean valid, final Consumer<T> validationErrorCallback) {
        if(!valid) {
            validationErrorCallback.accept(value);
        }
    }
}
