package ch.codeblock.qrinvoice.qrcode;

import ch.codeblock.qrinvoice.BaseException;
import ch.codeblock.qrinvoice.NotYetImplementedException;
import ch.codeblock.qrinvoice.OutputFormat;
import ch.codeblock.qrinvoice.TechnicalException;
import com.google.zxing.BarcodeFormat;
import com.google.zxing.EncodeHintType;
import com.google.zxing.WriterException;
import com.google.zxing.client.j2se.MatrixToImageConfig;
import com.google.zxing.client.j2se.MatrixToImageWriter;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.qrcode.QRCodeWriter;
import com.google.zxing.qrcode.decoder.ErrorCorrectionLevel;
import com.google.zxing.qrcode.encoder.Encoder;
import com.google.zxing.qrcode.encoder.QRCode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;

import static ch.codeblock.qrinvoice.QrInvoiceSpec.QR_CODE_LOGO_SIZE;
import static ch.codeblock.qrinvoice.QrInvoiceSpec.QR_CODE_SIZE;

/**
 * The following information comes from the specification v1.0
 * <p>"The measurements of the Swiss QR Code for printing must always be 46 x 46 mm (without surrounding quiet space) regardless of the Swiss QR Code version."</p>
 * <p>"To increase the recognizability and differentiation for users, the Swiss QR Code created for printout is to be overlaid with a Swiss cross logo measuring 7 x 7 mm."</p>
 */
public class SwissQrCodeWriter {
    private final Logger logger = LoggerFactory.getLogger(SwissQrCodeWriter.class);

    public byte[] write(final OutputFormat outputFormat, final String qrCodeString, final int desiredQrCodeSize) {
        try {
            final ByteArrayOutputStream baos = new ByteArrayOutputStream();
            renderSwissQrCode(outputFormat, qrCodeString, desiredQrCodeSize, baos);
            return baos.toByteArray();
        } catch (BaseException e) {
            throw e;
        } catch (Exception e) {
            throw new TechnicalException("Unexpected exception encountered during SwissQrCode creation", e);
        }
    }

    private void renderSwissQrCode(final OutputFormat outputFormat, final String qrCodeString, final int desiredQrCodeSize, final OutputStream outputStream) throws WriterException, IOException {
        switch (outputFormat) {
            case PNG:
                BufferedImage qrCodeImage = renderSwissQrCodeAsRasterizedGraphic(qrCodeString, desiredQrCodeSize);
                ImageIO.write(qrCodeImage, outputFormat.name(), outputStream);
                break;
            case PDF:
            default:
                throw new NotYetImplementedException("Output Format " + outputFormat + " has not yet been implemented");
        }
    }

    private BufferedImage renderSwissQrCodeAsRasterizedGraphic(final String qrCodeString, final int desiredQrCodeSize) throws WriterException, IOException {       // Create the ByteMatrix for the QR-Code that encodes the given String
        BufferedImage qrImage = renderQrCode(qrCodeString, desiredQrCodeSize);
        return overlayWithQrCodeLogo(qrImage);
    }

    private BufferedImage renderQrCode(final String qrCodeString, final int desiredQrCodeSize) throws WriterException {
        final Map<EncodeHintType, Object> hintMap = new HashMap<>();
        // error correction level M must be used according to the spec (4.1 - Error correction level - v1.0)
        hintMap.put(EncodeHintType.ERROR_CORRECTION, ErrorCorrectionLevel.M);
        // no margin, gets added later on in the image/rendering. is much easier because the spec says it has to be 5mm (2.5.3 - Swiss QR Code section - v1.0)
        hintMap.put(EncodeHintType.MARGIN, 0);
        final QRCodeWriter qrCodeWriter = new QRCodeWriter();
        final int minimalQrCodeSize = getMinimalQrCodeSize(qrCodeString, hintMap);
        final int optimalQrCodeRenderSize = getOptimalQrCodeRenderSize(desiredQrCodeSize, minimalQrCodeSize);

        if (logger.isTraceEnabled()) {
            logger.trace("desiredQrCodeSize: {}", desiredQrCodeSize);
            logger.trace("minimalQrCodeSize: {}", minimalQrCodeSize);
            logger.trace("optimalQrCodeRenderSize: {}", optimalQrCodeRenderSize);
        }

        final BitMatrix bitMatrix = qrCodeWriter.encode(qrCodeString, BarcodeFormat.QR_CODE, optimalQrCodeRenderSize, optimalQrCodeRenderSize, hintMap);
        final MatrixToImageConfig config = new MatrixToImageConfig(MatrixToImageConfig.BLACK, MatrixToImageConfig.WHITE);

        final BufferedImage qrImage = MatrixToImageWriter.toBufferedImage(bitMatrix, config);
        if (logger.isTraceEnabled()) {
            logger.trace("widht: {}", qrImage.getWidth());
            logger.trace("height: {}", qrImage.getHeight());
            logger.trace("qrImageWidth: {}", qrImage.getWidth());
            logger.trace("qrImageHeight: {}", qrImage.getHeight());
        }

        return qrImage;
    }

    private BufferedImage overlayWithQrCodeLogo(final BufferedImage qrImage) throws IOException {
        final int qrCodeWidth = qrImage.getWidth();
        final int qrCodeHeight = qrImage.getHeight();

        // Initialize combined image
        final BufferedImage combined = new BufferedImage(qrCodeWidth, qrCodeHeight, BufferedImage.TYPE_INT_ARGB);
        final Graphics2D g = (Graphics2D) combined.getGraphics();

        // Write QR code to new image at position 0/0
        g.drawImage(qrImage, 0, 0, null);
        g.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 1f));

        // Load logo image
        final int qrLogoWidth = Math.floorDiv(qrCodeWidth * QR_CODE_LOGO_SIZE.getWidth(), QR_CODE_SIZE.getWidth());
        final int qrLogoHeight = Math.floorDiv(qrCodeHeight * QR_CODE_LOGO_SIZE.getHeight(), QR_CODE_SIZE.getHeight());
        final Image logoImage = ImageIO.read(getClass().getResourceAsStream("/ch/codeblock/qrinvoice/standards/kreuz.png"))
                .getScaledInstance(qrLogoWidth,
                        qrLogoHeight,
                        -1 /*TODO check*/
                );

        // Calculate the delta height and width between QR code and logo
        final int deltaWidth = qrCodeWidth - qrLogoWidth;
        final int deltaHeight = qrCodeHeight - qrLogoHeight;

        // Write logo into combine image at position (deltaWidth / 2) and
        // (deltaHeight / 2). Background: Left/Right and Top/Bottom must be
        // the same space for the logo to be centered
        final int qrLogoPositionX = Math.round(deltaWidth / 2);
        final int qrLogoPositionY = Math.round(deltaHeight / 2);
        g.drawImage(logoImage, qrLogoPositionX, qrLogoPositionY, null);

        return combined;
    }

    private int getOptimalQrCodeRenderSize(final int desiredQrCodeSize, final int minimalQrCodeSize) {
        if (minimalQrCodeSize >= desiredQrCodeSize) {
            return minimalQrCodeSize;
        } else {
            return minimalQrCodeSize * Math.floorDiv(desiredQrCodeSize, minimalQrCodeSize);
        }
    }

    private int getMinimalQrCodeSize(final String qrCodeString, final Map<EncodeHintType, ?> hintMap) throws WriterException {
        // not the most efficient way, but we need to determine the minimal pixels required to display the qr code
        final QRCode qrCode = Encoder.encode(qrCodeString, ErrorCorrectionLevel.M, hintMap);
        return qrCode.getMatrix().getWidth();
    }
}
