package ch.codeblock.qrinvoice.util;

import java.math.BigInteger;
import java.util.stream.Collectors;

import static ch.codeblock.qrinvoice.util.StringUtils.removeWhitespaces;

/**
 * Util for Creditor Reference according to ISO 11649
 */
public final class CreditorReferenceUtils {
    private CreditorReferenceUtils(){}
    
    /** In the beginning there are two letters 'RF'*/
    private static final String PREFIX = "RF";
    /**at least four chars expected (RF + 2 Digits Checksum) */
    private static final int MIN_LENGTH = 4;
    /** The Creditor Reference is 25 characters long and alphanumeric*/
    private static final int MAX_LENGTH = 25;
    private static final BigInteger MOD_97 = new BigInteger("97");
    private static final BigInteger MOD_97_REMAINDER = BigInteger.ONE;

    /**
     * @param creditorReferenceInput
     * @return
     * @see <a href="https://en.wikipedia.org/wiki/Creditor_Reference">Creditor_Reference</a>
     */
    public static boolean isValid(final String creditorReferenceInput) {
        if(creditorReferenceInput == null) {
            return false;
        }
        
        // first, remove all whitespaces
        // --> RF45 1234 5123 45 -> RF451234512345
        final String creditorReference = removeWhitespaces(creditorReferenceInput);

        // 1. perform basic length an prefix checks
        final int length = creditorReference.length();
        if(MIN_LENGTH > length || length > MAX_LENGTH) {
            return false;
        }
        
        if(!creditorReference.startsWith(PREFIX)) {
            return  false;
        }
        
        // 2. Move the four initial characters to the end of the string
        // --> RF451234512345 -> 1234512345RF45
        final String creditorReferenceRearranged = creditorReference.substring(4) + creditorReference.substring(0, 4);

        // 3. Replace each letter in the string with two digits, thereby expanding the string, where A = 10, B = 11, ..., Z = 35.
        // --> 1234512345RF45 -> 1234512345271545
        final String numericCreditorReference = creditorReferenceRearranged.chars()
                .map(Character::getNumericValue)
                .mapToObj(String::valueOf)
                .collect(Collectors.joining());

        // 4. Interpret the string as a decimal integer and compute the remainder of that number on division by 97
        final BigInteger creditorReferenceNumber = new BigInteger(numericCreditorReference);
        // true and valid, if remainder equals 1
        return creditorReferenceNumber.mod(MOD_97).equals(MOD_97_REMAINDER);
    }
    

}
