package ch.codeblock.qrinvoice.util;

import ch.codeblock.qrinvoice.QrInvoiceSpec;

import java.math.BigInteger;
import java.util.stream.Collectors;

import static ch.codeblock.qrinvoice.util.StringUtils.removeWhitespaces;

public final class IbanUtils {
    private IbanUtils() {
    }

    private static final int CHARS_PER_BLOCK = 4;

    public static String formatIban(final String ibanInput) {
        if (ibanInput == null) {
            return "";
        }

        final String iban = removeWhitespaces(ibanInput);
        final StringBuilder ibanBuffer = new StringBuilder(iban);

        final int numberOfBlocks = iban.length() / CHARS_PER_BLOCK;
        for (int blockNr = 1; blockNr <= numberOfBlocks; blockNr++) {
            ibanBuffer.insert((blockNr * CHARS_PER_BLOCK) + blockNr - 1, ' ');
        }

        return ibanBuffer.toString();
    }

    private static final int MIN_LENGTH = 15;
    private static final int MAX_LENGTH = 34;
    private static final BigInteger MOD_97 = new BigInteger("97");
    private static final BigInteger MOD_97_REMAINDER = BigInteger.ONE;

    /**
     * @param ibanInput
     * @return
     * @see <a href="https://en.wikipedia.org/wiki/International_Bank_Account_Number#Validating_the_IBAN">Validating_the_IBAN</a>
     */
    public static boolean isValidIBAN(final String ibanInput, final boolean validateCountryCode) {
        if (ibanInput == null) {
            return false;
        }

        // first, remove all whitespaces
        final String iban = removeWhitespaces(ibanInput);

        // 1. Check that the total IBAN length is correct as per the country. If not, the IBAN is invalid
        // --> simple, non-country based check of the iban length
        final int len = iban.length();
        if (len < MIN_LENGTH || len > MAX_LENGTH) {
            return false;
        }

        // 2. Move the four initial characters to the end of the string
        // --> CH3908704016075473007 -> 08704016075473007CH39
        final String ibanRearranged = iban.substring(4) + iban.substring(0, 4);

        // 3. Replace each letter in the string with two digits, thereby expanding the string, where A = 10, B = 11, ..., Z = 35.
        final String numericIBAN = ibanRearranged.chars()
                .map(Character::getNumericValue)
                .mapToObj(String::valueOf)
                .collect(Collectors.joining());

        // 4. Interpret the string as a decimal integer and compute the remainder of that number on division by 97
        final BigInteger ibanNumber = new BigInteger(numericIBAN);
        final boolean ibanValid = ibanNumber.mod(MOD_97).equals(MOD_97_REMAINDER);

        // 5. validate country code
        if (ibanValid && validateCountryCode) {
            // only if IBAN is valid perform country code verifcation
            return validateCountryCode(iban);
        } else {
            // just return the result
            return ibanValid;
        }
    }

    private static boolean validateCountryCode(final String iban) {
        final String countryCode = iban.substring(0, 2);
        return QrInvoiceSpec.SUPPORTED_IBAN_COUNTRIES.contains(countryCode);
    }

    /** Start of the QR IID (Instituts-Identifikation) range - see https://www.iso-20022.ch/lexikon/qr-iban/ for further details */
    private static final int QR_IID_RANGE_START = 30000;
    /** End of the QR IID (Instituts-Identifikation) range - see https://www.iso-20022.ch/lexikon/qr-iban/ for further details */
    private static final int QR_IID_RANGE_END = 31999;
    
    // currently this can lead to validate the iban twice, maybe this should be refactory to parse the iban and work on an IBAN object to avoid double validation and parsing 
    public static boolean isQrIBAN(final String ibanInput) {
        if (ibanInput == null) {
            return false;
        }

        // first, remove all whitespaces
        final String iban = removeWhitespaces(ibanInput);
        if(!isValidIBAN(iban, false)) {
            return false;
        }
        
        // IBAN: CH3181239000001245689
        //
        // CH	2-stelliger Ländercode / CH für die Schweiz
        // 31	2-stellige Prüfziffer, pro Konto und Bank individuell
        // 81239	Bank-Clearing-Nummer bzw. IID oder QR-IID, zur eindeutigen Identifizierung der kontoführenden Bank des Zahlungsempfängers (pro Bank individuell)
        // 000001245689	12-stellige Kontonummer des Zahlungsempfängers, wo nötig mit führenden Nullen auf 12 Stellen ergänzt
        final int iid = Integer.parseInt(iban.substring(4, 9));
        
        // check that IID is within the special QR-IID range
        return (QR_IID_RANGE_START <= iid && iid <= QR_IID_RANGE_END);
    }
}
