package ch.codeblock.qrinvoice.util;

import static ch.codeblock.qrinvoice.util.StringUtils.removeWhitespaces;

/**
 * Util for QR-Reference (former ISR reference number)
 */
public final class QRReferenceUtils {
    private QRReferenceUtils(){}

    /** The QR Reference must have a length of 27 characters */
    private static final int LENGTH = 27;
    private static final int[] DIGIT_TABLE = { 0, 9, 4, 6, 8, 2, 7, 1, 3, 5 };
    
    /**
     * 
     * QR reference: 27 characters, numeric, check sum calculation according to Modulo 10 recursive (27th position of the reference)
     * 
     * @param qrReferenceInput
     * @return
     * @see <a href="https://en.wikipedia.org/wiki/Creditor_Reference">Creditor_Reference</a>
     */
    public static boolean isValid(final String qrReferenceInput) {
        if(qrReferenceInput == null) {
            return false;
        }
        
        // first, remove all whitespaces
        // --> 11 00012 34560 00000 00008 13457 -> 110001234560000000000813457
        final String qrReference = removeWhitespaces(qrReferenceInput);

        // 1. perform basic length an prefix checks
        if(LENGTH != qrReference.length()) {
            return false;
        }

        // 2. split ESR number to verify checksum digit
        // 110001234560000000000813457 -> 11000123456000000000081345
        final String qrReferenceWithoutChecksumDigit = qrReference.substring(0, 26);
        // 110001234560000000000813457 -> 7
        final int checksumDigit = Character.getNumericValue(qrReference.charAt(26));
        
        // 3. calculate checksum digit an verify it
        return modulo10Recurisve(qrReferenceWithoutChecksumDigit) == checksumDigit;
    }

    private static int modulo10Recurisve(final String nummer) {
        int transfer = 0;

        for (final char c : nummer.toCharArray()) {
            final int digit = Character.getNumericValue(c);
            transfer = DIGIT_TABLE[(transfer + digit) % 10];
        }

        return (10 - transfer) % 10;
    }
    
}
