package ch.codeblock.qrinvoice.layout;

import ch.codeblock.qrinvoice.util.NumberUtils;

public class Rect<T extends Number> extends Dimension<T> {

    /**
     * the lower left x-coordinate.
     */
    private final T llx;

    /**
     * the lower left y-coordinate.
     */
    private final T lly;

    /**
     * the upper right x-coordinate.
     */
    private final T urx;

    /**
     * the upper right y-coordinate.
     */
    private final T ury;

    /**
     * Constructs a <CODE>Rect</CODE> -object.
     *
     * @param llx lower left x
     * @param lly lower left y
     * @param urx upper right x
     * @param ury upper right y
     */
    public Rect(final T llx, final T lly, final T urx, final T ury) {
        this(llx, lly, urx, ury, NumberUtils.subtract(urx, llx), NumberUtils.subtract(ury, lly));
    }   
    
    /**
     * Constructs a <CODE>Rect</CODE> -object.
     *
     * @param llx lower left x
     * @param lly lower left y
     * @param urx upper right x
     * @param ury upper right y
     */
    public Rect(final T llx, final T lly, final T urx, final T ury, final T width, final T height) {
        super(width, height);
        this.llx = llx;
        this.lly = lly;
        this.urx = urx;
        this.ury = ury;
    }


    /**
     * Returns the lower left x-coordinate.
     *
     * @return the lower left x-coordinate
     */
    public T getLowerLeftX() {
        return llx;
    }

    /**
     * Returns the upper right x-coordinate.
     *
     * @return the upper right x-coordinate
     */
    public T getUpperRightX() {
        return urx;
    }


    /**
     * Returns the upper right y-coordinate.
     *
     * @return the upper right y-coordinate
     */
    public T getUpperRightY() {
        return ury;
    }


    /**
     * Returns the lower left y-coordinate.
     *
     * @return the lower left y-coordinate
     */
    public T getLowerLeftY() {
        return lly;
    }
    
    public Rect<T> move(final Point<T> offset) {
        return move(offset.getX(), offset.getY());
    }
    
    public Rect<T> move(final T offsetX, final T offsetY) {
        return Rect.createUsingDimension(NumberUtils.add(llx,offsetX), NumberUtils.add(lly,offsetY), getWidth(), getHeight());
    }

    public static <T extends Number> Rect<T> create(final T llx, final T lly, final T urx, final T ury) {
        return new Rect<>(llx, lly, urx, ury);
    }

    public static <T extends Number> Rect<T> createUsingDimension(final T llx, final T lly, final Dimension<T> dimension) {
        return createUsingDimension(llx,lly, dimension.getWidth(), dimension.getHeight());
    }
    
    public static <T extends Number> Rect<T> createUsingDimension(final T llx, final T lly, final T width, final T height) {
        return new Rect<>(llx, lly, NumberUtils.add(llx, width), NumberUtils.add(lly, height), width, height);
    }

}
