package ch.codeblock.qrinvoice.model;

import ch.codeblock.qrinvoice.model.interfaces.Address;
import ch.codeblock.qrinvoice.model.util.AddressUtils;

import java.util.Objects;

/**
 * <p>From the specification v1.0</p>
 * <table border="1" summary="Excerpt from the specification">
 * <tr><th>Language</th><th>General Definition</th><th>Field Definition</th></tr>
 * <tr><td>EN</td><td>Ultimate creditor<br>Ultimate creditor<br>Information about the ultimate creditor</td><td>Optional data group; may only be used in agreement with the creditor’s financial institution</td></tr>
 * <tr><td>DE</td><td>Endgültiger Zahlungsempfänger<br>Endgültiger Zahlungsempfänger<br>Informationen zum endgültigen Zahlungsempfänger</td><td>Optionale Datengruppe; darf nur in Absprache mit dem Finanzinstitut des Zahlungsempfängers verwendet werden</td></tr>
 * <tr><td>FR</td><td>Bénéficiaire final<br>Bénéficiaire final<br>Informations sur le bénéficiaire final</td><td>Groupe de données optionnel; ne peut être utilisé que d'entente avec l'établissement financier du bénéficiaire</td></tr>
 * <tr><td>IT</td><td>Beneficiario finale<br>Beneficiario finale<br>Informazioni sul beneficiario finale</td><td>Gruppo di dati opzionale<br>Può essere usato solo in accordo con l’istituto finanziario del beneficiario.</td></tr>
 * </table>
 * <p>Data Structure Element</p>
 * <pre>
 * QRCH
 * +UltmtCdtr
 * </pre>
 */
public class UltimateCreditor implements Address {
    /**
     * <p>From the specification v1.0</p>
     * <table border="1" summary="Excerpt from the specification">
     * <tr><th>Language</th><th>General Definition</th><th>Field Definition</th></tr>
     * <tr><td>EN</td><td>Name<br>Name or company of the ultimate creditor</td><td>Maximum 70 characters permitted. first name (optional, if available) and last name or company name.</td></tr>
     * <tr><td>DE</td><td>Name<br>Name bzw. Firma des endgültigen Zahlungsempfängers</td><td>Maximal 70 Zeichen zulässig; Vorname (optional, falls verfügbar) und Name oder Firmenbezeichnung</td></tr>
     * <tr><td>FR</td><td>Name<br>Nom ou entreprise du bénéficiaire final</td><td>70 caractères au maximum; prénom (optionnel, si disponible) et nom ou raison sociale</td></tr>
     * <tr><td>IT</td><td>Nome<br>Nome o azienda del beneficiario finale</td><td>Ammessi massimo 70 caratteri<br>Nome (opzionale, se disponibile) + cognome o ragione sociale</td></tr>
     * </table>
     * <p>Status: {@link Status#DEPENDENT}</p>
     * <p>Data Structure Element</p>
     * <pre>
     * QRCH
     * +UltmtCdtr
     * ++Name
     * </pre>
     */
    private String name;
    /**
     * <p>From the specification v1.0</p>
     * <table border="1" summary="Excerpt from the specification">
     * <tr><th>Language</th><th>General Definition</th><th>Field Definition</th></tr>
     * <tr><td>EN</td><td>Street<br>Street/P.O. box of the ultimate creditor</td><td>Maximum 70 characters permitted; may not include any house or building number.</td></tr>
     * <tr><td>DE</td><td>Strasse<br>Strasse/Postfach des endgültigen Zahlungsempfängers</td><td>Maximal 70 Zeichen zulässig; darf keine Haus- bzw. Gebäudenummer enthalten.</td></tr>
     * <tr><td>FR</td><td>Rue<br>Rue/Case postale du bénéficiaire final</td><td>70 caractères au maximum admis; ne peut pas contenir un numéro de maison ou de bâtiment.</td></tr>
     * <tr><td>IT</td><td>Via<br>Via/casella postale dell’indirizzo del beneficiario finale</td><td>Ammessi massimo 70 caratteri; non deve contenere numeri civici.</td></tr>
     * </table>
     * <p>Status: {@link Status#OPTIONAL}</p>
     * <p>Data Structure Element</p>
     * <pre>
     * QRCH
     * +UltmtCdtr
     * ++StrtNm
     * </pre>
     */
    private String streetName;
    /**
     * <p>From the specification v1.0</p>
     * <table border="1" summary="Excerpt from the specification">
     * <tr><th>Language</th><th>General Definition</th><th>Field Definition</th></tr>
     * <tr><td>EN</td><td>House number<br>House number of the ultimate creditor</td><td>Maximum 16 characters permitted</td></tr>
     * <tr><td>DE</td><td>Hausnummer<br>Hausnummer des endgültigen Zahlungsempfängers</td><td>Maximal 16 Zeichen zulässig</td></tr>
     * <tr><td>FR</td><td>Numéro de maison<br>Numéro de maison du bénéficiaire final</td><td>16 caractères au maximum admis</td></tr>
     * <tr><td>IT</td><td>Numero civico<br>Numero civico dell’indirizzo del beneficiario finale</td><td>Ammessi massimo 16 caratteri</td></tr>
     * </table>
     * <p>Status: {@link Status#OPTIONAL}</p>
     * <p>Data Structure Element</p>
     * <pre>
     * QRCH
     * +UltmtCdtr
     * ++BldgNb
     * </pre>
     */
    private String houseNumber;
    /**
     * <p>From the specification v1.0</p>
     * <table border="1" summary="Excerpt from the specification">
     * <tr><th>Language</th><th>General Definition</th><th>Field Definition</th></tr>
     * <tr><td>EN</td><td>Postal code<br>Postal code of the ultimate creditor</td><td>Maximum 16 characters permitted; is always to be entered without a country code prefix.</td></tr>
     * <tr><td>DE</td><td>Postleitzahl<br>Postleitzahl des endgültigen Zahlungsempfängers</td><td>Maximal 16 Zeichen zulässig; ist immer ohne vorangestellten Landescode anzugeben.</td></tr>
     * <tr><td>FR</td><td>Code postal<br>Code postal du bénéficiaire final</td><td>16 caractères au maximum admis; toujours à indiquer sans code de pays de tête.</td></tr>
     * <tr><td>IT</td><td>Numero postale di avviamento<br>Numero postale di avviamento dell’indirizzo del beneficiario finale</td><td>Ammessi massimo 16 caratteri; indicare sempre il numero postale di avviamento senza anteporre la sigla della nazione.</td></tr>
     * </table>
     * <p>Status: {@link Status#DEPENDENT}</p>
     * <p>Data Structure Element</p>
     * <pre>
     * QRCH
     * +UltmtCdtr
     * ++PstCd
     * </pre>
     */
    private String postalCode;
    /**
     * <p>From the specification v1.0</p>
     * <table border="1" summary="Excerpt from the specification">
     * <tr><th>Language</th><th>General Definition</th><th>Field Definition</th></tr>
     * <tr><td>EN</td><td>City<br>City of the ultimate creditor</td><td>Maximum 35 characters permitted</td></tr>
     * <tr><td>DE</td><td>Ort<br>Ort der endgültigen Zahlungsempfängers</td><td>Maximal 35 Zeichen zulässig</td></tr>
     * <tr><td>FR</td><td>Lieu<br>Lieu du bénéficiaire final</td><td>35 caractères au maximum admis</td></tr>
     * <tr><td>IT</td><td>Località<br>Località dell’indirizzo del beneficiario finale</td><td>Ammessi massimo 35 caratteri</td></tr>
     * </table>
     * <p>Status: {@link Status#DEPENDENT}</p>
     * <p>Data Structure Element</p>
     * <pre>
     * QRCH
     * +UltmtCdtr
     * ++TwnNm
     * </pre>
     */
    private String city;
    /**
     * <p>From the specification v1.0</p>
     * <table border="1" summary="Excerpt from the specification">
     * <tr><th>Language</th><th>General Definition</th><th>Field Definition</th></tr>
     * <tr><td>EN</td><td>Country<br>Country of the ultimate creditor</td><td>Two-digit country code according to ISO 3166-1</td></tr>
     * <tr><td>DE</td><td>Land<br>Land des endgültigen Zahlungsempfängers</td><td>Zweistelliger Landescode gemäss ISO 3166-1</td></tr>
     * <tr><td>FR</td><td>Pays<br>Pays du bénéficiaire final</td><td>Code de pays à deux positions selon ISO 3166-1</td></tr>
     * <tr><td>IT</td><td>Nazione<br>Nazione dell’indirizzo del beneficiario finale</td><td>Codice nazione a due caratteri secondo ISO 3166-1</td></tr>
     * </table>
     * <p>Status: {@link Status#DEPENDENT}</p>
     * <p>Data Structure Element</p>
     * <pre>
     * QRCH
     * +UltmtCdtr
     * ++Ctry
     * </pre>
     */
    private String country;

    @Override
    public String getName() {
        return name;
    }

    @Override
    public void setName(final String name) {
        this.name = name;
    }

    @Override
    public String getStreetName() {
        return streetName;
    }

    @Override
    public void setStreetName(final String streetName) {
        this.streetName = streetName;
    }

    @Override
    public String getHouseNumber() {
        return houseNumber;
    }

    @Override
    public void setHouseNumber(final String houseNumber) {
        this.houseNumber = houseNumber;
    }

    @Override
    public String getPostalCode() {
        return postalCode;
    }

    @Override
    public void setPostalCode(final String postalCode) {
        this.postalCode = postalCode;
    }

    @Override
    public void setPostalCode(final int postalCode) {
        setPostalCode(String.valueOf(postalCode));
    }

    @Override
    public String getCity() {
        return city;
    }

    @Override
    public void setCity(final String city) {
        this.city = city;
    }

    @Override
    public String getCountry() {
        return country;
    }

    @Override
    public void setCountry(final String country) {
        this.country = country;
    }

    @Override
    public boolean isEmpty() {
        return AddressUtils.isEmpty(this);
    }

    @Override
    public String toString() {
        return "UltimateCreditor{" +
                "name='" + name + '\'' +
                ", streetName='" + streetName + '\'' +
                ", houseNumber='" + houseNumber + '\'' +
                ", postalCode='" + postalCode + '\'' +
                ", city='" + city + '\'' +
                ", country='" + country + '\'' +
                '}';
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        final UltimateCreditor that = (UltimateCreditor) o;
        return Objects.equals(name, that.name) &&
                Objects.equals(streetName, that.streetName) &&
                Objects.equals(houseNumber, that.houseNumber) &&
                Objects.equals(postalCode, that.postalCode) &&
                Objects.equals(city, that.city) &&
                Objects.equals(country, that.country);
    }

    @Override
    public int hashCode() {
        return Objects.hash(name, streetName, houseNumber, postalCode, city, country);
    }
}

