package ch.codeblock.qrinvoice.model.builder;

import ch.codeblock.qrinvoice.QrInvoiceSpec;
import ch.codeblock.qrinvoice.model.PaymentAmountInformation;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Currency;
import java.util.Date;

public final class PaymentAmountInformationBuilder {
    private BigDecimal amount;
    private Currency currency;
    private LocalDate date;

    private PaymentAmountInformationBuilder() {
    }

    public static PaymentAmountInformationBuilder create() {
        return new PaymentAmountInformationBuilder();
    }

    public PaymentAmountInformationBuilder amount(BigDecimal amount) {
        this.amount = amount;
        return this;
    }
    
    public PaymentAmountInformationBuilder amount(final double amount) {
        return amount(BigDecimal.valueOf(amount));
    }

    public PaymentAmountInformationBuilder currency(Currency currency) {
        this.currency = currency;
        return this;
    }

    public PaymentAmountInformationBuilder date(LocalDate date) {
        this.date = date;
        return this;
    }

    public PaymentAmountInformationBuilder date(final Date date) {
        final LocalDate dateLocalDate = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        return date(dateLocalDate);
    }

    /**
     * Set a given amount in Swiss Francs
     * <pre>
     *     .chf(BigDecimal.valueOf(42.00))
     * </pre>
     * is the same as
     * <pre>
     *     .currency(Currency.getInstance("CHF"))
     *     .amount(BigDecimal.valueOf(42.00))
     * </pre>
     *
     * @param amount
     * @return
     */
    public PaymentAmountInformationBuilder chf(final BigDecimal amount) {
        return currency(QrInvoiceSpec.CHF).amount(amount);
    }

    /**
     * Set a given amount in Swiss Francs
     * <pre>
     *     .chf(42.00d)
     * </pre>
     * is the same as
     * <pre>
     *     .currency(Currency.getInstance("CHF"))
     *     .amount(42.00d)
     * </pre>
     *
     * @param amount
     * @return
     */
    public PaymentAmountInformationBuilder chf(final double amount) {
        return currency(QrInvoiceSpec.CHF).amount(amount);
    }


    /**
     * Set a given amount in Euro
     * <pre>
     *     .eur(BigDecimal.valueOf(42.00))
     * </pre>
     * is the same as
     * <pre>
     *     .currency(Currency.getInstance("EUR"))
     *     .amount(BigDecimal.valueOf(42.00))
     * </pre>
     *
     * @param amount
     * @return
     */
    public PaymentAmountInformationBuilder eur(final BigDecimal amount) {
        return currency(QrInvoiceSpec.EUR).amount(amount);
    }

    /**
     * Set a given amount in Euro
     * <pre>
     *     .eur(42.00d)
     * </pre>
     * is the same as
     * <pre>
     *     .currency(Currency.getInstance("EUR"))
     *     .amount(42.00d)
     * </pre>
     *
     * @param amount
     * @return
     */
    public PaymentAmountInformationBuilder eur(final double amount) {
        return currency(QrInvoiceSpec.EUR).amount(amount);
    }
    /**
     * Set a given amount in Euro
     * <pre>
     *     .eur()
     * </pre>
     * is the same as
     * <pre>
     *     .currency(Currency.getInstance("EUR"))
     * </pre>
     * 
     * @return 
     */
    public PaymentAmountInformationBuilder eur() {
        return currency(QrInvoiceSpec.EUR);
    }

    /**
     * Set a given amount in Euro
     * <pre>
     *     .chf()
     * </pre>
     * is the same as
     * <pre>
     *     .currency(Currency.getInstance("CHF"))
     * </pre>
     *
     * @return
     */
    public PaymentAmountInformationBuilder chf() {
        return currency(QrInvoiceSpec.CHF);
    }

    public PaymentAmountInformation build() {
        PaymentAmountInformation paymentAmountInformation = new PaymentAmountInformation();
        paymentAmountInformation.setAmount(amount);
        paymentAmountInformation.setCurrency(currency);
        paymentAmountInformation.setDate(date);
        return paymentAmountInformation;
    }
}
