package ch.codeblock.qrinvoice.model.builder;

import ch.codeblock.qrinvoice.QrInvoiceSpec;
import ch.codeblock.qrinvoice.model.*;
import ch.codeblock.qrinvoice.model.validation.QrInvoiceValidator;
import ch.codeblock.qrinvoice.model.validation.ValidationException;
import ch.codeblock.qrinvoice.model.validation.ValidationResult;
import ch.codeblock.qrinvoice.util.CollectionUtils;

import java.util.List;
import java.util.function.Consumer;

/**
 * <p>Create always a new QrInvoiceBuilder instance in order to create a new QrInvoice</p>
 * <p>This builder is not thread safe on purpose.</p>
 */
public final class QrInvoiceBuilder {
    // Builder delegates
    private CreditorInformationBuilder creditorInformationBuilder;
    private CreditorBuilder creditorBuilder;
    private UltimateCreditorBuilder ultimateCreditorBuilder;
    private UltimateDebtorBuilder ultimateDebtorBuilder;
    private PaymentAmountInformationBuilder paymentAmountInformationBuilder;
    private PaymentReferenceBuilder paymentReferenceBuilder;
    private AlternativeSchemesBuilder alternativeSchemesBuilder;

    // CreditorInformation
    private String iban;

    // AlternativeSchemeParameters
    private List<String> alternativeSchemeParameters;

    private QrInvoiceBuilder() {
    }

    public static QrInvoiceBuilder create() {
        return new QrInvoiceBuilder();
    }

    private CreditorInformationBuilder creditorInformation() {
        if (creditorInformationBuilder == null) {
            creditorInformationBuilder = CreditorInformationBuilder.create();
        }
        return creditorInformationBuilder;
    }

    public CreditorBuilder creditor() {
        if (creditorBuilder == null) {
            creditorBuilder = CreditorBuilder.create();
        }
        return creditorBuilder;
    }

    public UltimateCreditorBuilder ultimateCreditor() {
        if (ultimateCreditorBuilder == null) {
            ultimateCreditorBuilder = UltimateCreditorBuilder.create();
        }
        return ultimateCreditorBuilder;
    }

    public UltimateDebtorBuilder ultimateDebtor() {
        if (ultimateDebtorBuilder == null) {
            ultimateDebtorBuilder = UltimateDebtorBuilder.create();
        }
        return ultimateDebtorBuilder;
    }

    public PaymentAmountInformationBuilder paymentAmountInformation() {
        if (paymentAmountInformationBuilder == null) {
            paymentAmountInformationBuilder = PaymentAmountInformationBuilder.create();
        }
        return paymentAmountInformationBuilder;
    }

    public PaymentReferenceBuilder paymentReference() {
        if (paymentReferenceBuilder == null) {
            paymentReferenceBuilder = PaymentReferenceBuilder.create();
        }
        return paymentReferenceBuilder;
    }

    private AlternativeSchemesBuilder alternativeSchemes() {
        if (alternativeSchemesBuilder == null) {
            alternativeSchemesBuilder = AlternativeSchemesBuilder.create();
        }
        return alternativeSchemesBuilder;
    }

    public QrInvoiceBuilder creditor(final Consumer<CreditorBuilder> func) {
        func.accept(creditor());
        return this;
    }

    public QrInvoiceBuilder ultimateCreditor(final Consumer<UltimateCreditorBuilder> func) {
        func.accept(ultimateCreditor());
        return this;
    }

    public QrInvoiceBuilder ultimateDebtor(final Consumer<UltimateDebtorBuilder> func) {
        func.accept(ultimateDebtor());
        return this;
    }

    public QrInvoiceBuilder paymentAmountInformation(final Consumer<PaymentAmountInformationBuilder> func) {
        func.accept(paymentAmountInformation());
        return this;
    }

    public QrInvoiceBuilder paymentReference(final Consumer<PaymentReferenceBuilder> func) {
        func.accept(paymentReference());
        return this;
    }

    public QrInvoiceBuilder creditorIBAN(final String iban) {
        this.iban = iban;
        return this;
    }

    public QrInvoiceBuilder alternativeSchemeParameters(final List<String> alternativeSchemeParameters) {
        this.alternativeSchemeParameters = alternativeSchemeParameters;
        return this;
    }

    /**
     * 
     * @return the {@link QrInvoice}
     * @throws ValidationException 
     */
    public QrInvoice build() throws ValidationException {
        // create child elements
        final Header header = buildHeader();
        final CreditorInformation creditorInformation = buildCreditorInformation();
        final UltimateCreditor ultimateCreditor = buildUltimateCreditor();
        final UltimateDebtor ultimateDebtor = buildUltimateDebtor();
        final PaymentAmountInformation paymentAmountInformation = buildPaymentAmountInformation();
        final PaymentReference paymentReference = buildPaymentReference();
        final AlternativeSchemes alternativeSchemes = buildAlternativeSchemes();

        // build QrInvoice root object
        final QrInvoice qrInvoice = new QrInvoice();
        qrInvoice.setHeader(header);
        qrInvoice.setCreditorInformation(creditorInformation);
        qrInvoice.setUltimateCreditor(ultimateCreditor);
        qrInvoice.setUltimateDebtor(ultimateDebtor);
        qrInvoice.setPaymentAmountInformation(paymentAmountInformation);
        qrInvoice.setPaymentReference(paymentReference);
        qrInvoice.setAlternativeSchemes(alternativeSchemes);

        // perform validation
        QrInvoiceValidator.create().validate(qrInvoice).throwExceptionOnErrors();

        return qrInvoice;
    }

    private Header buildHeader() {
        return HeaderBuilder.create()
                .qrType(QrInvoiceSpec.QR_TYPE)
                .version(Short.parseShort(QrInvoiceSpec.VERSION))
                .codingType(QrInvoiceSpec.CODING_TPYE).build();
    }

    private CreditorInformation buildCreditorInformation() {
        return creditorInformation()
                .creditor(creditor().build())
                .iban(iban)
                .build();
    }

    private UltimateCreditor buildUltimateCreditor() {
        final UltimateCreditor ultimateCreditor;
        if (ultimateCreditorBuilder != null) {
            ultimateCreditor = ultimateCreditorBuilder.build();
        } else {
            ultimateCreditor = null;
        }
        return ultimateCreditor;
    }

    private UltimateDebtor buildUltimateDebtor() {
        final UltimateDebtor ultimateDebtor;
        if (ultimateDebtorBuilder != null) {
            ultimateDebtor = ultimateDebtorBuilder.build();
        } else {
            ultimateDebtor = null;
        }
        return ultimateDebtor;
    }

    private PaymentAmountInformation buildPaymentAmountInformation() {
        return paymentAmountInformation().build();
    }

    private PaymentReference buildPaymentReference() {
        return paymentReference().build();
    }

    private AlternativeSchemes buildAlternativeSchemes() {
        // optional group
        if (CollectionUtils.isNotEmpty(alternativeSchemeParameters)) {
            alternativeSchemes().alternativeSchemeParameters(alternativeSchemeParameters);
        }
        if (alternativeSchemesBuilder != null) {
            return alternativeSchemesBuilder.build();
        } else {
            return null;
        }
    }

}
