package ch.codeblock.qrinvoice.model.mapper;

import ch.codeblock.qrinvoice.MappingException;
import ch.codeblock.qrinvoice.model.*;
import ch.codeblock.qrinvoice.util.DecimalFormatFactory;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Currency;

public class ModelToSwissPaymentsCodeMapper {

    public SwissPaymentsCode map(QrInvoice qrInvoice) {
        final SwissPaymentsCode spc = new SwissPaymentsCode();

        mapHeader(qrInvoice.getHeader(), spc);
        mapCreditorInformation(qrInvoice.getCreditorInformation(), spc);
        mapUltimateCreditor(qrInvoice.getUltimateCreditor(), spc);
        mapPaymentAmountInformation(qrInvoice.getPaymentAmountInformation(), spc);
        mapUltimateDebtor(qrInvoice.getUltimateDebtor(), spc);
        mapPaymentReference(qrInvoice.getPaymentReference(), spc);
        mapAlternativeSchemes(qrInvoice.getAlternativeSchemes(), spc);

        return spc;
    }

    private void mapPaymentReference(final PaymentReference paymentReference, final SwissPaymentsCode spc) {
        if(paymentReference == null) {
            throw new MappingException("PaymentReference must be given");
        }
        spc.setTp(paymentReference.getReferenceType().getReferenceTypeCode());
        spc.setRef(paymentReference.getReference());
        spc.setUstrd(paymentReference.getUnstructuredMessage());
    }

    private void mapPaymentAmountInformation(final PaymentAmountInformation paymentAmountInformation, final SwissPaymentsCode spc) {
        if(paymentAmountInformation == null) {
            throw new MappingException("PaymentAmountInformation must be given");
        }
        
        final BigDecimal amount = paymentAmountInformation.getAmount();
        if(amount != null) {
            spc.setAmt(DecimalFormatFactory.createQrCodeAmountFormat().format(amount));
        } else {
            spc.setAmt(null); 
        }

        final Currency currency = paymentAmountInformation.getCurrency();
        if(currency!= null) {
            spc.setCcy(currency.getCurrencyCode());
        } else {
            spc.setCcy(null);
        }

        final LocalDate date = paymentAmountInformation.getDate();
        if(date != null) {
            spc.setReqdExctnDt(date.format(DateTimeFormatter.ISO_DATE));
        } else {
            spc.setReqdExctnDt(null);
        }
    }

    private void mapUltimateDebtor(final UltimateDebtor ultimateDebtor, final SwissPaymentsCode spc) {
        if(ultimateDebtor == null) {
            // optional
            return;
        }
        spc.setUdName(ultimateDebtor.getName());
        spc.setUdStrtNm(ultimateDebtor.getStreetName());
        spc.setUdBldgNb(ultimateDebtor.getHouseNumber());
        spc.setUdPstCd(ultimateDebtor.getPostalCode());
        spc.setUdTwnNm(ultimateDebtor.getCity());
        spc.setUdCtry(ultimateDebtor.getCountry());
    }

    private void mapHeader(final Header header, final SwissPaymentsCode spc) {
        if(header == null) {
            throw new MappingException("Header must be given");
        }
        spc.setQrType(header.getQrType());
        spc.setVersion(String.format("%04d", header.getVersion()));
        spc.setCoding(Byte.toString(header.getCodingType())); // fixed length 1
    }

    private void mapCreditorInformation(final CreditorInformation creditorInformation, final SwissPaymentsCode spc) {
        if(creditorInformation == null) {
            throw new MappingException("CreditorInformation must be given");
        }
        spc.setIban(creditorInformation.getIban());
        mapCreditor(creditorInformation.getCreditor(), spc);
    }

    private void mapCreditor(final Creditor creditor, final SwissPaymentsCode spc) {
        if(creditor == null) {
            throw new MappingException("Creditor must be given");
        }
        spc.setCrName(creditor.getName());
        spc.setCrStrtNm(creditor.getStreetName());
        spc.setCrBldgNb(creditor.getHouseNumber());
        spc.setCrPstCd(creditor.getPostalCode());
        spc.setCrTwnNm(creditor.getCity());
        spc.setCrCtry(creditor.getCountry());
    }

    private void mapUltimateCreditor(final UltimateCreditor ultimateCreditor, final SwissPaymentsCode spc) {
        if(ultimateCreditor == null) {
            // optional
            return;
        }
        spc.setUcrName(ultimateCreditor.getName());
        spc.setUcrStrtNm(ultimateCreditor.getStreetName());
        spc.setUcrBldgNb(ultimateCreditor.getHouseNumber());
        spc.setUcrPstCd(ultimateCreditor.getPostalCode());
        spc.setUcrTwnNm(ultimateCreditor.getCity());
        spc.setUcrCtry(ultimateCreditor.getCountry());
    }

    private void mapAlternativeSchemes(final AlternativeSchemes alternativeSchemes, final SwissPaymentsCode spc) {
        if(alternativeSchemes == null) {
            // optional
            return;
        }
        spc.setAltPmts(alternativeSchemes.getAlternativeSchemeParameters());
    }
    
}
