package ch.codeblock.qrinvoice.model.validation;

import ch.codeblock.qrinvoice.model.Creditor;
import ch.codeblock.qrinvoice.model.UltimateCreditor;
import ch.codeblock.qrinvoice.model.UltimateDebtor;
import ch.codeblock.qrinvoice.model.interfaces.Address;

import static ch.codeblock.qrinvoice.model.validation.ValidationUtils.*;
import static ch.codeblock.qrinvoice.util.CountryUtils.isValidIsoCode;

public class AddressValidator {
    public void validate(final Creditor creditor, final ValidationResult result) {
        validateAddress("creditorinformation.creditor", creditor, true, result);
    }

    public void validate(final UltimateCreditor ultimateCreditor, final ValidationResult result) {
        validateAddress("ultimateCreditor", ultimateCreditor, false, result);
    }

    public void validate(final UltimateDebtor ultimateDebtor, final ValidationResult result) {
        validateAddress("ultimateDebtor", ultimateDebtor, false, result);
    }

    private void validateAddress(final String baseDataPath, final Address address, final boolean mandatoryGroup, final ValidationResult result) {
        if (address == null) {
            if (mandatoryGroup) {
                result.addError(baseDataPath, null, "validation.error.address.group");
            }
            return;
        }
        
        // @formatter:off
        // lengths
        validateLength        (address.getName(),        1, 70, (value) -> result.addError(baseDataPath, "name",        value, "validation.error.address.name"));
        validateOptionalLength(address.getStreetName(),  1, 70, (value) -> result.addError(baseDataPath, "streetName",  value, "validation.error.address.streetName"));
        validateOptionalLength(address.getHouseNumber(), 1, 16, (value) -> result.addError(baseDataPath, "houseNumber", value, "validation.error.address.houseNumber"));
        validateLength        (address.getPostalCode(),  1, 16, (value) -> result.addError(baseDataPath, "postalCode",  value, "validation.error.address.postalCode"));
        validateLength        (address.getCity(),        1, 35, (value) -> result.addError(baseDataPath, "city",        value, "validation.error.address.city"));
        validateTrue(address.getCountry(), 
                isValidIsoCode(address.getCountry()),           (value) -> result.addError(baseDataPath, "country",     value, "validation.error.address.country"));
        
        // characters
        validateCharacters    (address.getName(),               (value) -> result.addError(baseDataPath, "name",        value, "validation.error.invalidCharacters"));
        validateCharacters    (address.getStreetName(),         (value) -> result.addError(baseDataPath, "streetName",  value, "validation.error.invalidCharacters"));
        validateCharacters    (address.getHouseNumber(),        (value) -> result.addError(baseDataPath, "houseNumber", value, "validation.error.invalidCharacters"));
        validateCharacters    (address.getPostalCode(),         (value) -> result.addError(baseDataPath, "postalCode",  value, "validation.error.invalidCharacters"));
        validateCharacters    (address.getCity(),               (value) -> result.addError(baseDataPath, "city",        value, "validation.error.invalidCharacters"));
        // @formatter:on 
    }

}
