package ch.codeblock.qrinvoice.model.validation;

import ch.codeblock.qrinvoice.util.CollectionUtils;

import java.util.*;
import java.util.regex.Pattern;

public class ValidationResult {
    private static final Pattern LINE_BREAK_PATTERN = Pattern.compile("\\r\\n?|\\n");
    private List<ValidationError> errors;

    public <T> void addError(final String parentDataPath, final String dataPath, final T value, final String... errorMsgKeys) {
        final String completeDataPath = ((dataPath != null) ? parentDataPath + '.' + dataPath : parentDataPath);
        addError(new ValidationError<>(completeDataPath, value, errorMsgKeys));
    }

    private void addError(final ValidationError error) {
        if (errors == null) {
            errors = new ArrayList<>();
        }
        errors.add(error);
    }

    public boolean isEmpty() {
        return CollectionUtils.isEmpty(errors);
    }

    public boolean hasErrors() {
        return !isEmpty();
    }

    /**
     * if there are {@link #getErrors()}, throw a {@link ValidationException}, otherwise it is a silent operation
     * 
     * @throws ValidationException 
     */
    public void throwExceptionOnErrors() throws ValidationException {
        if(hasErrors()) {
            throw new ValidationException(this);
        }
    }
    
    public List<ValidationError> getErrors() {
        return errors != null ? errors : Collections.emptyList();
    }

    public String getValidationErrorSummary() {
        if (hasErrors()) {
            final StringBuilder sb = new StringBuilder();
            sb.append("QrInvoice has validation errors:");
            sb.append(System.lineSeparator());

            // only english error messages
            final ResourceBundle labels = ResourceBundle.getBundle("qrinvoice", Locale.ENGLISH);

            int errorNr = 1;
            for (final ValidationError error : errors) {
                if (errorNr > 1) {
                    sb.append(System.lineSeparator());
                }
                sb.append(errorNr++).append(". ").append(error.getDataPath()).append('\'');
                sb.append(" has invalid value '").append(error.getValue()).append("'");

                Arrays.stream(error.getErrorMsgKeys())
                        .map(labels::getString)
                        .map(LINE_BREAK_PATTERN::split)
                        .flatMap(Arrays::stream)
                        .forEach(
                                s -> sb.append(System.lineSeparator()).append("    =>").append(" ").append(s)
                        );
            }

            return sb.toString();
        } else {
            return "No validation errors";
        }
    }

    @Override
    public String toString() {
        return "ValidationResult{" +
                "errors=" + errors +
                '}';
    }

    public class ValidationError<T> {
        private final String dataPath;
        private final T value;
        private final String[] errorMsgKeys;

        public ValidationError(final String dataPath, final T value, final String... errorMsgKeys) {
            this.dataPath = dataPath;
            this.value = value;
            this.errorMsgKeys = errorMsgKeys;
        }

        public String getDataPath() {
            return dataPath;
        }

        public T getValue() {
            return value;
        }

        public String[] getErrorMsgKeys() {
            return errorMsgKeys;
        }


        @Override
        public String toString() {
            return "ValidationError{" +
                    "dataPath='" + dataPath + '\'' +
                    ", value=" + value +
                    ", errorMsgKeys='" + Arrays.toString(errorMsgKeys) + '\'' +
                    '}';
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            final ValidationError<?> that = (ValidationError<?>) o;
            return Objects.equals(dataPath, that.dataPath) &&
                    Objects.equals(value, that.value) &&
                    Arrays.equals(errorMsgKeys, that.errorMsgKeys);
        }

        @Override
        public int hashCode() {
            return Objects.hash(dataPath, value, errorMsgKeys);
        }
    }
}
