package ch.codeblock.qrinvoice.paymentpart;

import ch.codeblock.qrinvoice.OutputFormat;
import ch.codeblock.qrinvoice.QrInvoiceSpec;
import ch.codeblock.qrinvoice.TechnicalException;
import ch.codeblock.qrinvoice.config.SystemProperties;
import ch.codeblock.qrinvoice.layout.LayoutException;
import ch.codeblock.qrinvoice.layout.Point;
import ch.codeblock.qrinvoice.layout.Rect;
import ch.codeblock.qrinvoice.model.PaymentAmountInformation;
import ch.codeblock.qrinvoice.model.PaymentReference;
import ch.codeblock.qrinvoice.model.QrInvoice;
import ch.codeblock.qrinvoice.output.PaymentPart;
import ch.codeblock.qrinvoice.util.DecimalFormatFactory;
import ch.codeblock.qrinvoice.util.IbanUtils;
import com.itextpdf.text.BaseColor;
import com.itextpdf.text.Chunk;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Font;
import com.itextpdf.text.Image;
import com.itextpdf.text.PageSize;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.SplitCharacter;
import com.itextpdf.text.pdf.BaseFont;
import com.itextpdf.text.pdf.ColumnText;
import com.itextpdf.text.pdf.PdfChunk;
import com.itextpdf.text.pdf.PdfContentByte;
import com.itextpdf.text.pdf.PdfWriter;

import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.time.LocalDate;
import java.util.function.Function;

import static ch.codeblock.qrinvoice.model.util.AddressUtils.toAddress;
import static com.itextpdf.text.Element.ALIGN_LEFT;
import static com.itextpdf.text.Utilities.millimetersToPoints;

public class IText5PaymentPartWriter extends AbstractItextPaymentPartWriter implements IPaymentPartWriter {
    private final Rectangle pageCanvas;
    /** root lower left coordinates of the payment part on the pageCanvas in order to position the elements */
    private final Point<Float> rootLowerLeft;
    private final Rectangle informationSectionRect;
    private BaseFont regularFont;
    private BaseFont boldFont;
    private Font headingFont;
    private Font valueFont;

    public IText5PaymentPartWriter(final PaymentPartWriterOptions options) {
        super(options);
        switch (options.getPageSize()) {
            case A4:
                pageCanvas = PageSize.A4;
                break;
            case A5:
                pageCanvas = PageSize.A5.rotate();
                break;
            case A6:
            default:
                pageCanvas = PageSize.A6.rotate();
                break;
        }
        // determine the lower left position of the payment part on the page canvas
        rootLowerLeft = new Point<>(pageCanvas.getWidth() - getPaymentPartWidth(), 0.0f);
        informationSectionRect = itextRectangle(INFORMATION_SECTION_RECT.move(rootLowerLeft));
        try {
            boldFont = BaseFont.createFont(QrInvoiceSpec.DEFAULT_FONT_NAME_BOLD, CHARSET, true);
            regularFont = BaseFont.createFont(QrInvoiceSpec.DEFAULT_FONT_NAME, CHARSET, true);
            headingFont = new Font(boldFont, FONT_SIZE_HEADING);
            valueFont = new Font(regularFont, FONT_SIZE_VALUE);
        } catch (Exception e) {
            throw new TechnicalException("Error while creating fonts", e);
        }

    }

    @Override
    public PaymentPart write(QrInvoice qrInvoice, final BufferedImage qrCodeImage) {
        try {
            final ByteArrayOutputStream baos = new ByteArrayOutputStream(25 * 1024);
            final Document document = new Document(pageCanvas, 0, 0, 0, 0);
            final PdfWriter writer = PdfWriter.getInstance(document, baos);
            document.open();

            addTitleSection(writer);

            addQrCodeImage(writer, qrCodeImage);

            addAmount(writer, qrInvoice.getPaymentAmountInformation());

            addInformationSection(writer, qrInvoice);

            drawPaymentPartBoundaryLines(writer.getDirectContent());

            document.close();
            return new PaymentPart(OutputFormat.PDF, baos.toByteArray());
        } catch (DocumentException | IOException e) {
            throw new TechnicalException("Unexpected exception occurred during the creation of the payment part", e);
        }
    }

    private void addTitleSection(PdfWriter writer) {
        final PdfContentByte content = writer.getDirectContent();
        content.beginText();

        final float x = rootLowerLeft.getX() + QUIET_SPACE_PTS;
        content.setFontAndSize(headingFont.getBaseFont(), FONT_SIZE_TITLE);
        content.showTextAligned(ALIGN_LEFT, getLabel("Title"), x, getPaymentPartHeight() - 20, 0);

        content.setFontAndSize(headingFont.getBaseFont(), FONT_SIZE_HEADING);
        content.showTextAligned(ALIGN_LEFT, getLabel("Subtitle"), x, getPaymentPartHeight() - 30, 0);

        content.setFontAndSize(regularFont, FONT_SIZE_VALUE);
        content.showTextAligned(ALIGN_LEFT, getLabel("Transfer"), x, getPaymentPartHeight() - 42, 0);

        content.endText();
    }

    private void addQrCodeImage(final PdfWriter writer, final BufferedImage qrCodeImage) throws IOException, DocumentException {
        final Image image = Image.getInstance(qrCodeImage, null);
        final Rect<Float> absolutePos = QR_CODE_RECTANGLE.move(rootLowerLeft);
        image.setAbsolutePosition(absolutePos.getLowerLeftX(), absolutePos.getLowerLeftY());
        final float size = QR_CODE_RECTANGLE.getWidth();
        image.scaleToFit(size, size);
        final PdfContentByte canvas = writer.getDirectContent();
        canvas.addImage(image);
    }


    private void addInformationSection(PdfWriter writer, final QrInvoice qrInvoice) throws DocumentException {
        if (System.getProperty(SystemProperties.DEBUG_PAYMENT_PART_LAYOUT) != null) {
            informationSectionRect.setBorder(Rectangle.BOX);
            informationSectionRect.setBorderWidth(PAYMENT_PART_BOUNDARY_LINE_WIDTH);
            informationSectionRect.setBorderColor(BaseColor.BLACK);
            writer.getDirectContent().rectangle(informationSectionRect);
        }

        final PdfContentByte cb = writer.getDirectContent();
        final ColumnText ct = new ColumnText(cb);
        ct.setSimpleColumn(informationSectionRect);
        ct.setLeading(FIXED_LEADINGS.get(getOptions().getLayout()), MULTIPLIED_LEADINGS.get(getOptions().getLayout()));

        // Account
        ct.addText(chunk(getLabel("CdtrInf.IBAN"), headingFont).setLineHeight(FONT_SIZE_HEADING) /* set specific line height to get rid of top-padding/margin*/);
        ct.addText(chunk("\n", valueFont));
        ct.addText(chunk(IbanUtils.formatIban(qrInvoice.getCreditorInformation().getIban()), valueFont));
        ct.addText(chunk("\n", valueFont));

        // Creditor
        ct.addText(chunk(getLabel("CdtrInf.Cdtr"), headingFont));
        ct.addText(chunk("\n", valueFont));
        toAddress(qrInvoice.getCreditorInformation().getCreditor()).forEach(addressLine -> {
            ct.addText(chunk(addressLine, valueFont));
            ct.addText(chunk("\n", valueFont));
        });

        // Ultimate Creditor
        if (qrInvoice.getUltimateCreditor() != null && !qrInvoice.getUltimateCreditor().isEmpty()) {
            ct.addText(chunk(getLabel("UltmtCdtr"), headingFont));
            ct.addText(chunk("\n", valueFont));
            toAddress(qrInvoice.getUltimateCreditor()).forEach(addressLine -> {
                ct.addText(chunk(addressLine, valueFont));
                ct.addText(chunk("\n", valueFont));
            });
        }


        final PaymentReference paymentReference = qrInvoice.getPaymentReference();
        if (paymentReference != null) {
            // Reference number
            if (paymentReference.getReferenceType() != null) {
                switch (paymentReference.getReferenceType()) {
                    case QR_REFERENCE:
                    case CREDITOR_REFERENCE:
                        ct.addText(chunk(getLabel("RmtInf.Ref"), headingFont));
                        ct.addText(chunk("\n", valueFont));
                        ct.addText(chunk(paymentReference.getReference(), valueFont));
                        ct.addText(chunk("\n", valueFont));
                        break;
                    case WITHOUT_REFERENCE:
                        break;
                }
            }

            // Additional information
            if (paymentReference.getUnstructuredMessage() != null) {
                ct.addText(chunk(getLabel("RmtInf.Ustrd"), headingFont));
                ct.addText(chunk("\n", valueFont));
                ct.addText(chunk(paymentReference.getUnstructuredMessage(), valueFont));
                ct.addText(chunk("\n", valueFont));
            }
        }


        // Debtor
        ct.addText(chunk(getLabel("UltmtDbtr"), headingFont));
        ct.addText(chunk("\n", valueFont));
        if (qrInvoice.getUltimateDebtor() == null || qrInvoice.getUltimateDebtor().isEmpty()) {
            ct.go();
            final float leftX = informationSectionRect.getLeft();
            final float upperY = ct.getYLine() - FIXED_LEADINGS.get(getOptions().getLayout());
            final float lowerY = upperY - DEBTOR_FIELD.getHeight();
            final Rect<Float> debtorField = DEBTOR_FIELD.toRectangle(leftX, lowerY);

            writeFreeTextBox(ct.getCanvas(), debtorField);
            ct.addText(new Chunk("\n", valueFont).setLineHeight(DEBTOR_FIELD.getHeight() + FIXED_LEADINGS.get(getOptions().getLayout())));
        } else {
            toAddress(qrInvoice.getUltimateDebtor()).forEach(addressLine -> {
                ct.addText(chunk(addressLine, valueFont));
                ct.addText(chunk("\n", valueFont));
            });
        }


        //  Due date
        final LocalDate date = qrInvoice.getPaymentAmountInformation().getDate();
        if (date != null) {
            ct.addText(chunk(getLabel("CcyAmtDate.ReqdExctnDt"), headingFont));
            ct.addText(chunk("\n", valueFont));
            ct.addText(chunk(date.format(DATE_FORMATTER), valueFont));
        }

        switch (ct.go(false)) {
            case ColumnText.NO_MORE_TEXT:
                // ok case
                break;
            case ColumnText.NO_MORE_COLUMN:
                // nok
                if (System.getProperty(SystemProperties.IGNORE_LAYOUT_ERRORS) == null) {
                    throw new LayoutException("Not all content could be printed to the information section");
                }
        }
    }


    private void addAmount(final PdfWriter writer, final PaymentAmountInformation paymentAmountInformation) {
        final float lowerY = QUIET_SPACE_PTS;
        final float upperY = lowerY + AMOUNT_FIELD.getHeight();

        final PdfContentByte canvas = writer.getDirectContent();

        // header
        canvas.setFontAndSize(headingFont.getBaseFont(), FONT_SIZE_HEADING);
        canvas.beginText();
        final float currencyX = rootLowerLeft.getX() + QUIET_SPACE_PTS;
        final float amountX = rootLowerLeft.getX() + QUIET_SPACE_PTS + millimetersToPoints(12);
        final float headerY = upperY + 5;
        canvas.showTextAligned(ALIGN_LEFT, getLabel("Currency"), currencyX, headerY, 0);
        canvas.showTextAligned(ALIGN_LEFT, getLabel("Amount"), amountX, headerY, 0);
        canvas.endText();

        // body
        canvas.setFontAndSize(valueFont.getBaseFont(), FONT_SIZE_VALUE);
        canvas.beginText();
        final float valuesY = upperY - 10;
        canvas.showTextAligned(ALIGN_LEFT, paymentAmountInformation.getCurrency().getCurrencyCode(), currencyX, valuesY, 0);
        if (paymentAmountInformation.getAmount() != null) {
            final String formattedAmount = DecimalFormatFactory.createPrintAmountFormat().format(paymentAmountInformation.getAmount());
            canvas.showTextAligned(ALIGN_LEFT, formattedAmount, amountX, valuesY, 0);
        }
        canvas.endText();

        if (paymentAmountInformation.getAmount() == null) {
            final Rect<Float> rect = AMOUNT_FIELD.toRectangle(amountX, lowerY);
            writeFreeTextBox(canvas, rect);
        }
    }

    private void writeFreeTextBox(final PdfContentByte canvas, final Rect<Float> rect) {
        final float cornerLength = BOX_CORNER_LINE_LENGTH;
        final float leftX = rect.getLowerLeftX();
        final float lowerY = rect.getLowerLeftY();
        final float rightX = rect.getUpperRightX();
        final float upperY = rect.getUpperRightY();
        canvas.setColorStroke(BaseColor.BLACK);
        canvas.setLineWidth(PAYMENT_PART_BOUNDARY_LINE_WIDTH);

        // left bottom
        canvas.moveTo(leftX + cornerLength, lowerY);
        canvas.lineTo(leftX , lowerY);
        canvas.lineTo(leftX, lowerY + cornerLength);

        // left top
        canvas.moveTo(leftX + cornerLength, upperY);
        canvas.lineTo(leftX, upperY);
        canvas.lineTo(leftX, upperY - cornerLength);

        // right upper
        canvas.moveTo(rightX - cornerLength, upperY);
        canvas.lineTo(rightX, upperY);
        canvas.lineTo(rightX, upperY - cornerLength);

        // right bottom
        canvas.moveTo(rightX - cornerLength, lowerY);
        canvas.lineTo(rightX, lowerY);
        canvas.lineTo(rightX, lowerY + cornerLength);
    }

    private void drawPaymentPartBoundaryLines(final PdfContentByte canvas) {
        if (getOptions().isPaymentPartBoundaryLines()) {
            final float llx = rootLowerLeft.getX();
            final float lly = rootLowerLeft.getY();
            final float ulx = llx;
            final float uly = lly + getPaymentPartHeight();
            final float urx = llx + getPaymentPartWidth();
            final float ury = uly;

            canvas.moveTo(llx, lly);
            canvas.setColorStroke(BaseColor.BLACK);
            canvas.setLineWidth(PAYMENT_PART_BOUNDARY_LINE_WIDTH);

            canvas.lineTo(ulx, uly);
            canvas.moveTo(ulx, uly);
            canvas.lineTo(urx, ury);
            canvas.closePathStroke();
        }
    }

    private Chunk chunk(final String text, final Font font) {
        final Chunk chunk = new Chunk(text, font);
        final Function<String, Float> widthFunction = (str) -> font.getBaseFont().getWidthPoint(str, font.getSize());
        applyOptimalLineSplitting(text, informationSectionRect.getWidth(), widthFunction, () -> chunk.setSplitCharacter(SPLIT_ON_ALL_CHARACTERS));
        return chunk;
    }

    private Rectangle itextRectangle(final Rect<Float> rect) {
        return new Rectangle(rect.getLowerLeftX(), rect.getLowerLeftY(), rect.getUpperRightX(), rect.getUpperRightY());
    }

    private static final SplitOnAllCharacters SPLIT_ON_ALL_CHARACTERS = new SplitOnAllCharacters();

    private static final class SplitOnAllCharacters implements SplitCharacter {
        @Override
        public boolean isSplitCharacter(final int start, final int current, final int end, final char[] cc, final PdfChunk[] ck) {
            return true;
        }
    }
}
